using UnityEngine;
using UnityEngine.InputSystem;

public class NejikoController : MonoBehaviour
{
    const int MinLane = -2;
    const int MaxLane = 2;
    const float LaneWidth = 1.0f;
    const int DefaultLife = 3;
    const float StunDuration = 0.5f;

    public float Gravity;
    public float SpeedZ;
    public float SpeedX;
    public float SpeedJump;
    public float AccelerationZ;

    CharacterController _controller;
    Animator _animator;
    Vector3 _moveDirection = Vector3.zero;
    int _targetLane;
    int _life = DefaultLife;
    float _recoverTime = 0.0f;


    public int Life()
    {
        return _life;
    }

    bool IsStun()
    {
        return _recoverTime > 0.0f || _life <= 0;
    }

    void Start()
    {
        // 必要なコンポーネントを自動取得
        _controller = GetComponent<CharacterController>();
        _animator = GetComponent<Animator>();
    }

    void Update()
    {
        // デバッグ用
        if (Keyboard.current.leftArrowKey.wasPressedThisFrame) MoveToLeft();
        if (Keyboard.current.rightArrowKey.wasPressedThisFrame) MoveToRight();
        if (Keyboard.current.spaceKey.wasPressedThisFrame) Jump();

        if (IsStun())
        {
            // 動きを止め気絶状態からの復帰カウントを進める
            _moveDirection.x = 0.0f;
            _moveDirection.z = 0.0f;
            _recoverTime -= Time.deltaTime;
        }
        else
        {
            // 徐々に加速しZ方向に常に前進させる
            float acceleratedZ = _moveDirection.z + (AccelerationZ * Time.deltaTime);
            _moveDirection.z = Mathf.Clamp(acceleratedZ, 0, SpeedZ);

            // X方向は目標のポジションまでの差分の割合で速度を計算
            float ratioX = (_targetLane * LaneWidth - transform.position.x) / LaneWidth;
            _moveDirection.x = ratioX * SpeedX;
        }

        // 重力分の力を毎フレーム追加
        _moveDirection.y -= Gravity * Time.deltaTime;

        // 移動実行
        Vector3 globalDirection = transform.TransformDirection(_moveDirection);
        _controller.Move(globalDirection * Time.deltaTime);

        // 移動後接地してたらY方向の速度はリセットする
        if (_controller.isGrounded) _moveDirection.y = 0;

        // 速度が0以上なら走っているフラグをtrueにする
        _animator.SetBool("run", _moveDirection.z > 0.0f);
    }

    // 左のレーンに移動を開始
    public void MoveToLeft()
    {
        if (IsStun()) return;
        if (_controller.isGrounded && _targetLane > MinLane) _targetLane--;
    }

    // 右のレーンに移動を開始
    public void MoveToRight()
    {
        if (IsStun()) return;
        if (_controller.isGrounded && _targetLane < MaxLane) _targetLane++;
    }

    public void Jump()
    {
        if (IsStun()) return;
        if (_controller.isGrounded)
        {
            _moveDirection.y = SpeedJump;

            // ジャンプトリガーを設定
            _animator.SetTrigger("jump");
        }
    }

    // CharacterControllerに衝突判定が生じたときの処理
    void OnControllerColliderHit(ControllerColliderHit hit)
    {
        if (IsStun()) return;
        if (hit.gameObject.tag == "Robo")
        {
            // ライフを減らして気絶状態に移行
            _life--;
            _recoverTime = StunDuration;

            // ダメージトリガーを設定
            _animator.SetTrigger("damage");

            // ヒットしたオブジェクトは削除
            Destroy(hit.gameObject);
        }
    }
}